/** @file   tprinter.h
 * @brief   Declaraction of TPrinter - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_TPRINTER_H
#define H_ENG2D_TPRINTER_H

#include "eng2d_dll.h"
#include "bmfont.h"

namespace eng2d {

/** @class  TPrinter
 * @brief   Provides text printing services.
 * @author  Tomi Lamminsaari
 *
 * 
 */
class DLLIMPORT TPrinter
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Prints the given text on given bitmap with given font.
   *
   * This method can be used for calculating the line length by passing a
   * null-pointer as target bitmap.
   * @param     pB                Pointer to target bitmap. This can be null
   *                              pointer.
   * @param     pF                Pointer to font we'd like to use.
   * @param     rTxt              The text being printed
   * @param     pX                Pointer to variable telling the X-coordinate.
   *                              The variable at the pointer will be modified
   *                              so it tells the ending position.
   * @param     pY                Pointer to variable holdint the Y-coordinate.
   *                              The variable at the pointer will be modified
   *                              so it tells the ending position.
   */
  void print( BITMAP* pB, const GlyphSet* pF, const std::string& rTxt,
              int* pX, int* pY );


  /** Possible aliging of the text.
   */
  enum Align {
    /** Text will be aligned to the left. */
    ALIGN_LEFT,
    /** Text will be centered */
    ALIGN_CENTER,
    /** Text will be aligned to the right */
    ALIGN_RIGHT
  };
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	TPrinter();
	
	/** Constructor. The given bitmap is the target bitmap.
   * @param     pB                Pointer to target bitmap.
   */
  TPrinter( BITMAP* pB );


	/** Destructor
   */
	virtual ~TPrinter();

	/** Copy constructor.
   * @param     rO                Reference to another TPrinter
   */
  TPrinter( const TPrinter& rO );

	/** Assignment operator
   * @param     rO                Reference to another TPrinter
   * @return    Reference to us.
   */
  TPrinter& operator = ( const TPrinter& rO );



  ///
  /// Methods
  /// =======

  /** Prints the given text. Use @c pos(...) - method to set the printing
   * position.
   * @param     txt               The string being printed.
   */
  void print( const std::string& txt );
  
  /** Prints the given integer. Use @c pos(...) - method to set the
   * printing position.
   * @param     number            The number being printed.
   */
  void print( int number );
  

  /** Sets the next printing position
   * @param     x                 X-coordinate
   * @param     y                 Y-coordinate
   */
  void pos( int x, int y );

  /** Sets the target bitmap
   * @param     pB                Pointer to target bitmap.
   */
  void bitmap( BITMAP* pB );
  
  /** Sets the font.
   * @param     pF                Pointer to font
   * @param     siz               Size of the font.
   */
  void font( const BMFont* pF, int siz );
  
  /** Sets the font
   * @param     pF                Pointer to font
   * @param     id                ID of the font
   */
  void font( const BMFont* pF, const std::string& id );
  
  /** Sets the aligning
   * @param     a                 New aligning.
   */
  void align( Align a );
  

  ///
  /// Getter methods
  /// ==============

  /** Returns the length of given text in pixels when printed with current
   * font.
   * @param     txt               The text
   * @return    Length of the text in pixels.
   */
  int textLength( const std::string& txt );
  
  /** Returns the length of the given integer in pixels when printed with
   * current font.
   * @param     number            The integer
   * @return    Length of integer in pixels.
   */
  int textLength( int number );
  

protected:

  ///
  /// Members
  /// =======

  /** Pointer to target bitmap */
  BITMAP* m_pTarget;
  
  /** Pointer to font we'd like to use. */
  const GlyphSet* m_pF;
  
  /** The x coordinate we print next */
  int   m_x;
  /** The Y coordinate we print next */
  int   m_y;
  
  /** Aligning */
  Align m_align;

  /** This buffer is used when we convert the numbers to strings. */
  static char  s_convertBuffer[33];

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
